#ifndef __BRUTE_FORCE_SUDOKU_SOLVER__
#define __BRUTE_FORCE_SUDOKU_SOLVER__
/* This file declares the BruteForceSudokuSolver class, which solves a Sudoku puzzle
 * using a brute force algorithm.  The algorithm came from some example code in C on
 * Wikipedia (http://en.wikipedia.org/wiki/Algorithmics_of_sudoku).
 *
 * Date       User              Note
 * 2009-11-04 Eric Oulashin     Created
 * 2009-11-05 Eric Oulashin     Added mNumSolutions and mSolutions.
 * 2009-11-06 Eric Oulashin     Added reset().
 */

#include "SudokuSolution.h"

class BruteForceSudokuSolver
{
	public:
		// Constructor
		BruteForceSudokuSolver();

		// Desturctor
		~BruteForceSudokuSolver();

		// Sets a value in the puzzle grid.
		//
		// Parameters:
		//  pRow: The row index (0-based)
		//  pCol: The column index (0-based)
		//  pVal: The value to set at the cell
		void setVal(int pRow, int pCol, int pVal);

		// Returns a value from the puzzle grid.
		//
		// Parameters:
		//  pRow: The row index (0-based)
		//  pCol: The column index (0-based)
		//
		// Return value: The value at the given location.  Will return -1 if
		//               pRow or pCol are invalid.
		int getVal(int pRow, int pCol) const;

		// Returns the number of puzzle solutions.
		int NumSolutions() const;

		// Returns a value from one of the solved puzzles.
		//
		// Parameters:
		//  pPuzzleIndex: The index of the solved puzzle
		//  pRow: The row index (0-based)
		//  pCol: The column index (0-based)
		//
		// Return value: The value at the given location.  Will return -1 if
		//               pPuzzleIndex, pRow, or pCol are invalid.
		int getSolvedVal(int pPuzzleIndex, int pRow, int pCol) const;

		// Solves the puzzle using a brute force algorithm.
		void solve();

		// Resets the puzzle data.
		void reset();

		// Returns the number of numbers that have been set in the puzzle.  The intent of
		// this function is to be used before solving the puzzle to make sure there are
		// enough numbers to solve the puzzle - There must be at least 17 numbers set in
		// the puzzle before it can be solved.
		int numGivenNumbers() const;

	private:
		int mPuzzle[9][9];          // The Sudoku puzzle to be solved
		int mNumSolutions;          // The number of solutions to the puzzle
		SudokuSolution *mSolutions; // A pointer to an array of solutions to the puzzle

		// Solves the puzzle using a brute force algorithm.
		//
		// Parameters:
		//  pRow: The row index (0-based)
		//  pCol: The column index (0-based)
		void solve(int pRow, int pCol);

		// Returns whether or not it's safe to place a value at a given puzzle cell.
		//
		// Parameters:
		//  pRow: The row index (0-based)
		//  pCol: The column index (0-based)
		//  pVal: The value to test
		bool valSafe(int pRow, int pCol, int pVal) const;
};

#endif